/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using Inet.Viewer.Data;
using System;
using System.Collections.Generic;
namespace Inet.Viewer
{
    /// <summary>
    /// Enum to differ the different buttons for the toolbar
    /// </summary>
    public enum ToolbarButtonType
    {
        /// <summary>
        /// Button group of all "general" buttons
        /// </summary>
        General = 0,

        /// <summary>
        /// Button group of all "navigation" buttons such as "next page", "previous page", etc.
        /// </summary>
        Navigation = 1,

        /// <summary>
        /// Button group of all "view" buttons such as "double page view", "single page view", etc.
        /// </summary>
        View = 2,

        /// <summary>
        /// Button group of all "zoom" buttons, that is - "zoom in", "zoom out".
        /// </summary>
        Zoom = 3,

        /// <summary>
        /// Button group of all various "report" buttons such as "search", "export", and "refresh".
        /// </summary>
        Report = 4,

        /// <summary>
        /// Button responsible for the print action
        /// </summary>
        Print = 5,

        /// <summary>
        /// Button responsible for the refresh action
        /// </summary>
        Refresh = 6,

        /// <summary>
        /// Button responsible for the export action
        /// </summary>
        Export = 7,

        /// <summary>
        /// Button responsible for the search action
        /// </summary>
        Search = 8,

        /// <summary>
        /// Button responsible for showing the info dialog
        /// </summary>
        Info = 9,

        /// <summary>
        /// Button responsible for action loading java report archive (JRA)
        /// </summary>
        JRA_Laod = 10,

        /// <summary>
        /// Button responsible for action saving java report archive (JRA)
        /// </summary>
        JRA_Save = 11
    }

    /// <summary>
    /// The toolbar which contains various buttons.
    /// </summary>
    public interface IToolBar
    {
        /// <summary>
        /// Sets a button or a button group visible or invisible. </summary>
        /// <param name="buttonType"> The identifier for the button or button group </param>
        /// <param name="visible"> True if the button shoulb be visible otherwise false </param>
        /// <seealso cref= "Inet.Viewer.ToolbarButtonType"/>
        void SetButtonsVisible(ToolbarButtonType buttonType, bool visible);

        /// <summary>
        /// Return true if the button or button group is visible otherwise false. </summary>
        /// <param name="buttonType"> The specified button or button group </param>
        /// <returns> true if it is visible otherwise false </returns>
        /// <seealso cref= "Inet.Viewer.ToolbarButtonType"/>
        bool IsButtonsVisible(ToolbarButtonType buttonType);

        /// <summary>
        /// Shows or hides this component depending on the value of parameter <code>b</code>. </summary>
        /// <param> if <code>true</code>, shows this component; otherwise, hides this component</param>
        bool Visible { set; get; }

        /// <summary>
        /// The toolbar will be shown according to this ReportView
        /// </summary>
        IReportView ReportView { get; set; }

        /// <summary>
        /// Checks or unchecks to search button of this toolbar.
        /// </summary>
        bool SearchChecked { set; }

        /// <summary>
        /// Adds a progress to this toolbar.
        /// </summary>
        /// <param name="progress">the progress to add</param>
        void AddProgress(Progress progress);
        
        /// <summary>
        /// Selectes any running progress that match the specified filter delegate.
        /// </summary>
        /// <param name="filter">if the filter returns true for a given progress, the progress will be returned</param>
        IEnumerable<Progress> SelectProgress(Func<Progress, bool> filter);

        /// <summary>
        /// Increases the zoom level of the current viewer.
        /// </summary>
        void ZoomIn();

        /// <summary>
        /// Decreases the zoom level of the current viewer.
        /// </summary>
        void ZoomOut();
    }
}